<?php
/**
 * Main plugin class
 *
 * @package SIHA_WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Main SIHA_WooCommerce Class
 */
final class SIHA_WooCommerce {
    /**
     * The single instance of the class.
     *
     * @var SIHA_WooCommerce
     */
    protected static $_instance = null;

    /**
     * Main SIHA_WooCommerce Instance.
     *
     * Ensures only one instance of SIHA_WooCommerce is loaded or can be loaded.
     *
     * @return SIHA_WooCommerce Main instance.
     */
    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * SIHA_WooCommerce Constructor.
     */
    public function __construct() {
        $this->define_constants();
        $this->includes();
        $this->init_hooks();
    }

    /**
     * Define Constants.
     */
    private function define_constants() {
        $this->define( 'SIHA_WC_ABSPATH', dirname( SIHA_WC_PLUGIN_FILE ) . '/' );
        $this->define( 'SIHA_WC_PLUGIN_BASENAME', plugin_basename( SIHA_WC_PLUGIN_FILE ) );
    }

    /**
     * Define constant if not already set.
     *
     * @param string      $name  Constant name.
     * @param string|bool $value Constant value.
     */
    private function define( $name, $value ) {
        if ( ! defined( $name ) ) {
            define( $name, $value );
        }
    }

    /**
     * Include required core files used in admin and on the frontend.
     */
    public function includes() {
        // Core classes
        include_once SIHA_WC_ABSPATH . 'includes/class-siha-api.php';
        include_once SIHA_WC_ABSPATH . 'includes/class-siha-admin.php';
        include_once SIHA_WC_ABSPATH . 'includes/class-siha-product-sync.php';
        include_once SIHA_WC_ABSPATH . 'includes/class-siha-order-sync.php';
        
        // Helper functions
        include_once SIHA_WC_ABSPATH . 'includes/siha-helper-functions.php';
        
        // REST API
        if ( $this->is_request( 'rest' ) ) {
            include_once SIHA_WC_ABSPATH . 'includes/api/class-siha-rest-api.php';
        }
    }

    /**
     * Hook into actions and filters.
     */
    private function init_hooks() {
        // Load plugin text domain
        add_action( 'init', array( $this, 'load_plugin_textdomain' ) );
        
        // Initialize admin
        if ( is_admin() ) {
            add_action( 'admin_init', array( $this, 'admin_init' ) );
        }
    }

    /**
     * Load Localization files.
     */
    public function load_plugin_textdomain() {
        load_plugin_textdomain(
            'siha-woocommerce',
            false,
            dirname( plugin_basename( SIHA_WC_PLUGIN_FILE ) ) . '/languages/'
        );
    }

    /**
     * Initialize admin.
     */
    public function admin_init() {
        // Add settings link to plugins page
        add_filter( 'plugin_action_links_' . plugin_basename( SIHA_WC_PLUGIN_FILE ), array( $this, 'add_settings_link' ) );
    }

    /**
     * Add settings link to plugin actions.
     *
     * @param  array $actions Plugin action links.
     * @return array
     */
    public function add_settings_link( $actions ) {
        $settings = array(
            'settings' => sprintf(
                '<a href="%s">%s</a>',
                admin_url( 'admin.php?page=wc-settings&tab=siha' ),
                __( 'Settings', 'siha-woocommerce' )
            ),
        );
        return array_merge( $settings, $actions );
    }

    /**
     * What type of request is this?
     *
     * @param  string $type admin, ajax, cron or frontend.
     * @return bool
     */
    private function is_request( $type ) {
        switch ( $type ) {
            case 'admin':
                return is_admin();
            case 'ajax':
                return defined( 'DOING_AJAX' );
            case 'cron':
                return defined( 'DOING_CRON' );
            case 'frontend':
                return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
            case 'rest':
                return defined( 'REST_REQUEST' );
        }
        return false;
    }

    /**
     * Activation hook.
     */
    public static function activate() {
        if ( ! current_user_can( 'activate_plugins' ) ) {
            return;
        }
        
        // Add any activation logic here
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Deactivation hook.
     */
    public static function deactivate() {
        if ( ! current_user_can( 'activate_plugins' ) ) {
            return;
        }
        
        // Add any deactivation logic here
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Get the plugin url.
     *
     * @return string
     */
    public function plugin_url() {
        return untrailingslashit( plugins_url( '/', SIHA_WC_PLUGIN_FILE ) );
    }

    /**
     * Get the plugin path.
     *
     * @return string
     */
    public function plugin_path() {
        return untrailingslashit( plugin_dir_path( SIHA_WC_PLUGIN_FILE ) );
    }

    /**
     * Get the template path.
     *
     * @return string
     */
    public function template_path() {
        return apply_filters( 'siha_wc_template_path', 'siha/' );
    }
}
