<?php
/**
 * Product Synchronization
 *
 * Handles product synchronization between WooCommerce and SIHA.
 *
 * @package SIHA_WooCommerce
 */

defined( 'ABSPATH' ) || exit;

/**
 * SIHA_Product_Sync Class
 */
class SIHA_Product_Sync {
    /**
     * The single instance of the class.
     *
     * @var SIHA_Product_Sync
     */
    protected static $_instance = null;

    /**
     * API handler instance.
     *
     * @var SIHA_API
     */
    private $api;

    /**
     * Main SIHA_Product_Sync Instance.
     *
     * @return SIHA_Product_Sync Main instance.
     */
    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor.
     */
    public function __construct() {
        $this->api = new SIHA_API();
        
        // Product hooks
        add_action( 'save_post_product', array( $this, 'sync_product_on_save' ), 20, 3 );
        add_action( 'woocommerce_update_product', array( $this, 'sync_product' ), 10, 1 );
        add_action( 'woocommerce_new_product', array( $this, 'sync_product' ), 10, 1 );
        
        // Product meta
        add_action( 'woocommerce_product_options_sku', array( $this, 'add_product_meta_fields' ) );
        add_action( 'woocommerce_process_product_meta', array( $this, 'save_product_meta_fields' ), 20, 2 );
    }

    /**
     * Sync product when saved.
     *
     * @param int      $post_id Post ID.
     * @param WP_Post  $post    Post object.
     * @param bool     $update  Whether this is an existing post being updated.
     */
    public function sync_product_on_save( $post_id, $post, $update ) {
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        if ( 'product' !== $post->post_type ) {
            return;
        }

        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        $product = wc_get_product( $post_id );
        if ( $product ) {
            $this->sync_product( $product, ! $update );
        }
    }

    /**
     * Sync product with SIHA.
     *
     * @param WC_Product $product Product object.
     * @param bool       $is_new  Whether this is a new product.
     * @return array|WP_Error
     */
    public function sync_product( $product, $is_new = false ) {
        if ( 'no' === get_option( 'woocommerce_siha_auto_sync_products', 'yes' ) ) {
            return new WP_Error( 'auto_sync_disabled', __( 'Auto-sync is disabled for products.', 'siha-woocommerce' ) );
        }

        $product_data = $this->prepare_product_data( $product );
        if ( is_wp_error( $product_data ) ) {
            return $product_data;
        }

        $siha_product_id = $this->get_siha_product_id( $product->get_id() );
        
        if ( $siha_product_id ) {
            $result = $this->api->update_product( $siha_product_id, $product_data );
            if ( ! is_wp_error( $result ) ) {
                do_action( 'siha_product_updated', $product->get_id(), $result );
            }
            return $result;
        } else {
            $result = $this->api->create_product( $product_data );
            if ( ! is_wp_error( $result ) && isset( $result['id'] ) ) {
                $this->save_siha_product_id( $product->get_id(), $result['id'] );
                do_action( 'siha_product_created', $product->get_id(), $result );
            }
            return $result;
        }
    }

    /**
     * Prepare product data for SIHA API.
     *
     * @param WC_Product $product Product object.
     * @return array|WP_Error
     */
    private function prepare_product_data( $product ) {
        if ( ! $product ) {
            return new WP_Error( 'invalid_product', __( 'Invalid product.', 'siha-woocommerce' ) );
        }

        $data = array(
            'name'           => $product->get_name(),
            'description'    => $product->get_description(),
            'sku'            => $product->get_sku(),
            'price'          => $product->get_price(),
            'stock_quantity' => $product->get_stock_quantity(),
            'stock_status'   => $product->get_stock_status(),
            'weight'         => $product->get_weight(),
            'categories'     => $this->get_product_categories( $product ),
            'images'         => $this->get_product_images( $product ),
            'status'         => $product->get_status() === 'publish' ? 'active' : 'draft',
        );

        return apply_filters( 'siha_prepare_product_data', $data, $product );
    }

    /**
     * Get product categories.
     *
     * @param WC_Product $product Product object.
     * @return array
     */
    private function get_product_categories( $product ) {
        $categories = array();
        $terms = get_the_terms( $product->get_id(), 'product_cat' );
        
        if ( $terms && ! is_wp_error( $terms ) ) {
            foreach ( $terms as $term ) {
                $categories[] = array(
                    'id'   => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                );
            }
        }
        
        return $categories;
    }

    /**
     * Get product images.
     *
     * @param WC_Product $product Product object.
     * @return array
     */
    private function get_product_images( $product ) {
        $images = array();
        
        if ( $featured_image_id = $product->get_image_id() ) {
            $images[] = array(
                'id'  => $featured_image_id,
                'src' => wp_get_attachment_image_url( $featured_image_id, 'full' ),
                'alt' => get_post_meta( $featured_image_id, '_wp_attachment_image_alt', true ),
            );
        }
        
        return $images;
    }

    /**
     * Get SIHA product ID.
     *
     * @param int $product_id WooCommerce product ID.
     * @return string|false
     */
    private function get_siha_product_id( $product_id ) {
        return get_post_meta( $product_id, '_siha_product_id', true );
    }

    /**
     * Save SIHA product ID.
     *
     * @param int    $product_id WooCommerce product ID.
     * @param string $siha_id   SIHA product ID.
     * @return bool
     */
    private function save_siha_product_id( $product_id, $siha_id ) {
        return update_post_meta( $product_id, '_siha_product_id', sanitize_text_field( $siha_id ) );
    }

    /**
     * Add product meta fields.
     */
    public function add_product_meta_fields() {
        woocommerce_wp_checkbox( array(
            'id'          => '_siha_sync_enabled',
            'label'       => __( 'Sync with SIHA', 'siha-woocommerce' ),
            'description' => __( 'Enable synchronization with SIHA', 'siha-woocommerce' ),
            'value'       => get_post_meta( get_the_ID(), '_siha_sync_enabled', true ) ?: 'yes',
        ) );
    }

    /**
     * Save product meta fields.
     *
     * @param int     $post_id Post ID.
     * @param WP_Post $post    Post object.
     */
    public function save_product_meta_fields( $post_id, $post ) {
        $sync_enabled = isset( $_POST['_siha_sync_enabled'] ) ? 'yes' : 'no';
        update_post_meta( $post_id, '_siha_sync_enabled', $sync_enabled );
    }
}
