<?php
/**
 * Helper functions for SIHA WooCommerce Integration
 *
 * @package SIHA_WooCommerce
 */

defined( 'ABSPATH' ) || exit;

/**
 * Get plugin settings.
 *
 * @param string $key     Setting key.
 * @param mixed  $default Default value.
 * @return mixed
 */
function siha_get_setting( $key, $default = '' ) {
    $settings = get_option( 'woocommerce_siha_settings', array() );
    return isset( $settings[ $key ] ) ? $settings[ $key ] : $default;
}

/**
 * Check if debug mode is enabled.
 *
 * @return bool
 */
function siha_is_debug_mode() {
    return 'yes' === siha_get_setting( 'debug_mode', 'no' );
}

/**
 * Log a message to the WooCommerce log.
 *
 * @param string $message Message to log.
 * @param string $level   Log level (emergency, alert, critical, error, warning, notice, info, debug).
 */
function siha_log( $message, $level = 'info' ) {
    if ( ! siha_is_debug_mode() && 'error' !== $level ) {
        return;
    }

    $logger = wc_get_logger();
    $context = array( 'source' => 'siha-woocommerce' );
    
    if ( is_array( $message ) || is_object( $message ) ) {
        $message = print_r( $message, true );
    }
    
    $logger->log( $level, $message, $context );
}

/**
 * Get product categories as a formatted string.
 *
 * @param WC_Product $product Product object.
 * @param string     $sep     Separator between categories.
 * @return string
 */
function siha_get_product_categories( $product, $sep = ', ' ) {
    $terms = get_the_terms( $product->get_id(), 'product_cat' );
    
    if ( is_wp_error( $terms ) || empty( $terms ) ) {
        return '';
    }
    
    $categories = wp_list_pluck( $terms, 'name' );
    return implode( $sep, $categories );
}

/**
 * Format price for display.
 *
 * @param float  $price   Price to format.
 * @param string $context Context (view or db).
 * @return string
 */
function siha_format_price( $price, $context = 'view' ) {
    if ( 'db' === $context ) {
        return wc_format_decimal( $price, 2 );
    }
    
    return wc_price( $price );
}

/**
 * Get product image URL.
 *
 * @param WC_Product $product Product object.
 * @param string     $size    Image size.
 * @return string
 */
function siha_get_product_image_url( $product, $size = 'woocommerce_thumbnail' ) {
    $image_id = $product->get_image_id();
    
    if ( $image_id ) {
        $image = wp_get_attachment_image_src( $image_id, $size );
        return $image ? $image[0] : '';
    }
    
    return wc_placeholder_img_src( $size );
}

/**
 * Check if the current request is a REST API request.
 *
 * @return bool
 */
function siha_is_rest_api_request() {
    if ( empty( $_SERVER['REQUEST_URI'] ) ) {
        return false;
    }
    
    $rest_prefix = trailingslashit( rest_get_url_prefix() );
    $is_rest = ( false !== strpos( $_SERVER['REQUEST_URI'], $rest_prefix ) );
    
    return apply_filters( 'siha_is_rest_api_request', $is_rest );
}

/**
 * Get the current user's IP address.
 *
 * @return string
 */
function siha_get_ip_address() {
    if ( isset( $_SERVER['HTTP_X_REAL_IP'] ) ) {
        return sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_REAL_IP'] ) );
    } elseif ( isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
        // Proxy servers can send through this header like this: X-Forwarded-For: client1, proxy1, proxy2
        $ip_addresses = explode( ',', $_SERVER['HTTP_X_FORWARDED_FOR'] );
        return sanitize_text_field( trim( $ip_addresses[0] ) );
    } elseif ( isset( $_SERVER['REMOTE_ADDR'] ) ) {
        return sanitize_text_field( $_SERVER['REMOTE_ADDR'] );
    }
    
    return '';
}

/**
 * Format date for display.
 *
 * @param string $date   Date string.
 * @param string $format Date format.
 * @return string
 */
function siha_format_date( $date, $format = '' ) {
    if ( ! $format ) {
        $format = get_option( 'date_format' ) . ' ' . get_option( 'time_format' );
    }
    
    return date_i18n( $format, strtotime( $date ) );
}

/**
 * Get the current WooCommerce currency symbol.
 *
 * @return string
 */
function siha_get_currency_symbol() {
    return get_woocommerce_currency_symbol();
}

/**
 * Check if WooCommerce is active.
 *
 * @return bool
 */
function siha_is_woocommerce_active() {
    return class_exists( 'WooCommerce' );
}

/**
 * Check if a plugin is active.
 *
 * @param string $plugin Plugin basename.
 * @return bool
 */
function siha_is_plugin_active( $plugin ) {
    if ( ! function_exists( 'is_plugin_active' ) ) {
        include_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
    
    return is_plugin_active( $plugin );
}

/**
 * Get the plugin URL.
 *
 * @param string $path Optional. Path relative to the plugin URL.
 * @return string
 */
function siha_plugin_url( $path = '' ) {
    return untrailingslashit( plugins_url( $path, SIHA_WC_PLUGIN_FILE ) );
}

/**
 * Get the plugin path.
 *
 * @param string $path Optional. Path relative to the plugin directory.
 * @return string
 */
function siha_plugin_path( $path = '' ) {
    return untrailingslashit( plugin_dir_path( SIHA_WC_PLUGIN_FILE ) ) . ( $path ? '/' . $path : '' );
}

/**
 * Get template part.
 *
 * @param string $slug Template slug.
 * @param string $name Template name (default: '').
 */
function siha_get_template_part( $slug, $name = '' ) {
    $template = '';
    $plugin_path = siha_plugin_path( 'templates/' );
    
    // Look in yourtheme/slug-name.php and yourtheme/woocommerce-slug/name.php
    if ( $name ) {
        $template = locate_template( array( "{$slug}-{$name}.php", WC()->template_path() . "{$slug}-{$name}.php" ) );
    }
    
    // Get default slug-name.php from plugin
    if ( ! $template && $name && file_exists( $plugin_path . "{$slug}-{$name}.php" ) ) {
        $template = $plugin_path . "{$slug}-{$name}.php";
    }
    
    // If template file doesn't exist, look in yourtheme/slug.php and yourtheme/woocommerce-slug.php
    if ( ! $template ) {
        $template = locate_template( array( "{$slug}.php", WC()->template_path() . "{$slug}.php" ) );
    }
    
    // Allow 3rd party plugins to filter template file from their plugin.
    $template = apply_filters( 'siha_get_template_part', $template, $slug, $name );
    
    if ( $template ) {
        load_template( $template, false );
    }
}

/**
 * Get template.
 *
 * @param string $template_name Template name.
 * @param array  $args          Arguments (default: array()).
 * @param string $template_path Template path (default: '').
 * @param string $default_path  Default path (default: '').
 */
function siha_get_template( $template_name, $args = array(), $template_path = '', $default_path = '' ) {
    if ( ! empty( $args ) && is_array( $args ) ) {
        extract( $args ); // @codingStandardsIgnoreLine
    }
    
    $located = siha_locate_template( $template_name, $template_path, $default_path );
    
    if ( ! file_exists( $located ) ) {
        /* translators: %s template */
        _doing_it_wrong( __FUNCTION__, sprintf( __( '%s does not exist.', 'siha-woocommerce' ), '<code>' . $located . '</code>' ), '1.0.0' );
        return;
    }
    
    // Allow 3rd party plugin filter template file from their plugin.
    $located = apply_filters( 'siha_get_template', $located, $template_name, $args, $template_path, $default_path );
    
    do_action( 'siha_before_template_part', $template_name, $template_path, $located, $args );
    
    include $located;
    
    do_action( 'siha_after_template_part', $template_name, $template_path, $located, $args );
}

/**
 * Locate a template and return the path for inclusion.
 *
 * @param string $template_name Template name.
 * @param string $template_path Template path (default: '').
 * @param string $default_path  Default path (default: '').
 * @return string
 */
function siha_locate_template( $template_name, $template_path = '', $default_path = '' ) {
    if ( ! $template_path ) {
        $template_path = WC()->template_path();
    }
    
    if ( ! $default_path ) {
        $default_path = siha_plugin_path() . '/templates/';
    }
    
    // Look within passed path within the theme - this is priority.
    $template = locate_template(
        array(
            trailingslashit( $template_path ) . $template_name,
            $template_name,
        )
    );
    
    // Get default template/
    if ( ! $template ) {
        $template = $default_path . $template_name;
    }
    
    // Return what we found.
    return apply_filters( 'siha_locate_template', $template, $template_name, $template_path );
}
