<?php
/**
 * SIHA API Handler
 *
 * Handles all API communication with the SIHA platform.
 *
 * @package SIHA_WooCommerce
 */

defined( 'ABSPATH' ) || exit;

/**
 * SIHA_API Class
 */
class SIHA_API {
    /**
     * API Base URL
     *
     * @var string
     */
    private $api_url = 'https://api.siha.com/v1';

    /**
     * API Key
     *
     * @var string
     */
    private $api_key = '';

    /**
     * Debug mode
     *
     * @var bool
     */
    private $debug = false;

    /**
     * Constructor.
     */
    public function __construct() {
        $this->init_settings();
    }

    /**
     * Initialize settings.
     */
    private function init_settings() {
        $settings = get_option( 'woocommerce_siha_settings', array() );
        
        $this->api_key = ! empty( $settings['api_key'] ) ? $settings['api_key'] : '';
        $this->debug   = isset( $settings['debug_mode'] ) && 'yes' === $settings['debug_mode'];
    }

    /**
     * Make a request to the SIHA API.
     *
     * @param string $endpoint API endpoint.
     * @param string $method   HTTP method.
     * @param array  $data     Request data.
     * @return array|WP_Error
     */
    public function request( $endpoint, $method = 'GET', $data = array() ) {
        $url = trailingslashit( $this->api_url ) . ltrim( $endpoint, '/' );
        
        $args = array(
            'method'  => $method,
            'headers' => $this->get_headers(),
            'timeout' => 30,
        );

        if ( ! empty( $data ) ) {
            if ( 'GET' === $method ) {
                $url = add_query_arg( $data, $url );
            } else {
                $args['body'] = wp_json_encode( $data );
            }
        }

        $this->log( 'API Request: ' . $method . ' ' . $url );
        if ( 'GET' !== $method && ! empty( $data ) ) {
            $this->log( 'Request Data: ' . wp_json_encode( $data ) );
        }

        $response = wp_remote_request( $url, $args );
        
        if ( is_wp_error( $response ) ) {
            $this->log( 'API Error: ' . $response->get_error_message() );
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $body          = json_decode( wp_remote_retrieve_body( $response ), true );
        
        $this->log( 'Response Code: ' . $response_code );
        $this->log( 'Response Body: ' . wp_json_encode( $body ) );

        if ( $response_code >= 200 && $response_code < 300 ) {
            return $body;
        } else {
            $error_message = isset( $body['message'] ) ? $body['message'] : 'Unknown error occurred';
            $this->log( 'API Error: ' . $error_message );
            return new WP_Error( 'siha_api_error', $error_message, array( 'status' => $response_code ) );
        }
    }

    /**
     * Get API headers.
     *
     * @return array
     */
    private function get_headers() {
        return array(
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json',
            'Authorization' => 'Bearer ' . $this->api_key,
        );
    }

    /**
     * Log messages.
     *
     * @param string $message Message to log.
     */
    private function log( $message ) {
        if ( $this->debug ) {
            $logger = wc_get_logger();
            $logger->debug( $message, array( 'source' => 'siha-woocommerce' ) );
        }
    }

    /**
     * Authenticate with SIHA API.
     *
     * @param string $email    User email.
     * @param string $password User password.
     * @return array|WP_Error
     */
    public function authenticate( $email, $password ) {
        return $this->request( '/auth/vendor/login', 'POST', array(
            'email'    => $email,
            'password' => $password,
        ));
    }

    /**
     * Create a new product in SIHA.
     *
     * @param array $product_data Product data.
     * @return array|WP_Error
     */
    public function create_product( $product_data ) {
        return $this->request( '/products', 'POST', $product_data );
    }

    /**
     * Update a product in SIHA.
     *
     * @param string $product_id   SIHA product ID.
     * @param array  $product_data Product data.
     * @return array|WP_Error
     */
    public function update_product( $product_id, $product_data ) {
        return $this->request( '/products/' . $product_id, 'PUT', $product_data );
    }

    /**
     * Delete a product from SIHA.
     *
     * @param string $product_id SIHA product ID.
     * @return array|WP_Error
     */
    public function delete_product( $product_id ) {
        return $this->request( '/products/' . $product_id, 'DELETE' );
    }

    /**
     * Create a new order in SIHA.
     *
     * @param array $order_data Order data.
     * @return array|WP_Error
     */
    public function create_order( $order_data ) {
        return $this->request( '/orders', 'POST', $order_data );
    }

    /**
     * Update order status in SIHA.
     *
     * @param string $order_id     SIHA order ID.
     * @param string $status       New status.
     * @param string $status_note  Status note.
     * @return array|WP_Error
     */
    public function update_order_status( $order_id, $status, $status_note = '' ) {
        return $this->request( "/orders/{$order_id}/status", 'PUT', array(
            'status'      => $status,
            'status_note' => $status_note,
        ));
    }

    /**
     * Get inventory levels from SIHA.
     *
     * @param array $params Query parameters.
     * @return array|WP_Error
     */
    public function get_inventory( $params = array() ) {
        return $this->request( '/inventory', 'GET', $params );
    }

    /**
     * Update inventory levels in SIHA.
     *
     * @param array $inventory_data Inventory data.
     * @return array|WP_Error
     */
    public function update_inventory( $inventory_data ) {
        return $this->request( '/inventory/bulk-update', 'POST', $inventory_data );
    }
}
