<?php
/**
 * Admin settings for SIHA WooCommerce Integration
 *
 * @package SIHA_WooCommerce
 */

defined( 'ABSPATH' ) || exit;

/**
 * SIHA_Admin Class
 */
class SIHA_Admin {
    /**
     * The single instance of the class.
     *
     * @var SIHA_Admin
     */
    protected static $_instance = null;

    /**
     * Main SIHA_Admin Instance.
     *
     * @return SIHA_Admin Main instance.
     */
    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor.
     */
    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_menu_item' ) );
        add_filter( 'woocommerce_get_sections_products', array( $this, 'add_section' ) );
        add_filter( 'woocommerce_get_settings_products', array( $this, 'get_settings' ), 10, 2 );
        add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
    }

    /**
     * Add menu item.
     */
    public function add_menu_item() {
        add_submenu_page(
            'woocommerce',
            __( 'SIHA Settings', 'siha-woocommerce' ),
            __( 'SIHA', 'siha-woocommerce' ),
            'manage_woocommerce',
            'admin.php?page=wc-settings&tab=products&section=siha',
            '',
            50
        );
    }

    /**
     * Add settings section to the Products tab.
     *
     * @param array $sections Sections.
     * @return array
     */
    public function add_section( $sections ) {
        $sections['siha'] = __( 'SIHA Integration', 'siha-woocommerce' );
        return $sections;
    }

    /**
     * Get settings array.
     *
     * @param array  $settings Settings array.
     * @param string $current_section Current section.
     * @return array
     */
    public function get_settings( $settings, $current_section ) {
        if ( 'siha' === $current_section ) {
            $settings_siha = array();

            // Add Title to the Settings
            $settings_siha[] = array(
                'name' => __( 'SIHA Integration', 'siha-woocommerce' ),
                'type' => 'title',
                'desc' => __( 'The following options are used to configure the SIHA integration.', 'siha-woocommerce' ),
                'id'   => 'siha_integration',
            );

            // API Key
            $settings_siha[] = array(
                'name'     => __( 'API Key', 'siha-woocommerce' ),
                'desc_tip' => __( 'Enter your SIHA API key. You can find this in your SIHA vendor dashboard.', 'siha-woocommerce' ),
                'id'       => 'woocommerce_siha_api_key',
                'type'     => 'password',
                'desc'     => sprintf( __( 'Get your API key from %sSIHA Vendor Dashboard%s', 'siha-woocommerce' ), '<a href="https://vendor.siha.com/settings/api" target="_blank">', '</a>' ),
            );

            // Sync Settings
            $settings_siha[] = array(
                'name' => __( 'Synchronization', 'siha-woocommerce' ),
                'type' => 'title',
                'desc' => '',
                'id'   => 'siha_sync_settings',
            );

            // Auto Sync Products
            $settings_siha[] = array(
                'title'    => __( 'Auto Sync Products', 'siha-woocommerce' ),
                'desc'     => __( 'Enable automatic product synchronization', 'siha-woocommerce' ),
                'id'       => 'woocommerce_siha_auto_sync_products',
                'default'  => 'yes',
                'type'     => 'checkbox',
                'desc_tip' => __( 'When enabled, products will be automatically synced between WooCommerce and SIHA.', 'siha-woocommerce' ),
            );

            // Auto Sync Orders
            $settings_siha[] = array(
                'title'    => __( 'Auto Sync Orders', 'siha-woocommerce' ),
                'desc'     => __( 'Enable automatic order synchronization', 'siha-woocommerce' ),
                'id'       => 'woocommerce_siha_auto_sync_orders',
                'default'  => 'yes',
                'type'     => 'checkbox',
                'desc_tip' => __( 'When enabled, orders will be automatically synced to SIHA.', 'siha-woocommerce' ),
            );

            // Debug Mode
            $settings_siha[] = array(
                'title'    => __( 'Debug Mode', 'siha-woocommerce' ),
                'desc'     => __( 'Enable debug logging', 'siha-woocommerce' ),
                'id'       => 'woocommerce_siha_debug_mode',
                'default'  => 'no',
                'type'     => 'checkbox',
                'desc_tip' => __( 'Log SIHA API requests and responses for debugging.', 'siha-woocommerce' ),
            );

            // Manual Sync Section
            $settings_siha[] = array(
                'name' => __( 'Manual Sync', 'siha-woocommerce' ),
                'type' => 'title',
                'desc' => '',
                'id'   => 'siha_manual_sync',
            );

            // Manual Sync Buttons
            $settings_siha[] = array(
                'type' => 'siha_sync_buttons',
            );

            // End section
            $settings_siha[] = array(
                'type' => 'sectionend',
                'id'   => 'siha_integration',
            );

            return $settings_siha;
        }

        return $settings;
    }

    /**
     * Generate Sync Buttons HTML.
     *
     * @param string $value Field value.
     * @param array  $data  Field data.
     * @return string
     */
    public function generate_siha_sync_buttons_html( $value, $data ) {
        ob_start();
        ?>
        <tr valign="top">
            <th scope="row" class="titledesc">
                <label><?php esc_html_e( 'Manual Sync', 'siha-woocommerce' ); ?></label>
            </th>
            <td class="forminp">
                <div class="siha-sync-buttons">
                    <button type="button" id="siha-sync-products" class="button button-secondary">
                        <?php esc_html_e( 'Sync Products', 'siha-woocommerce' ); ?>
                    </button>
                    <button type="button" id="siha-sync-orders" class="button button-secondary">
                        <?php esc_html_e( 'Sync Orders', 'siha-woocommerce' ); ?>
                    </button>
                    <button type="button" id="siha-sync-inventory" class="button button-secondary">
                        <?php esc_html_e( 'Sync Inventory', 'siha-woocommerce' ); ?>
                    </button>
                    <p class="description">
                        <?php esc_html_e( 'Manually trigger synchronization between WooCommerce and SIHA.', 'siha-woocommerce' ); ?>
                    </p>
                </div>
            </td>
        </tr>
        <?php
        return ob_get_clean();
    }

    /**
     * Enqueue admin scripts and styles.
     */
    public function admin_enqueue_scripts() {
        $screen    = get_current_screen();
        $screen_id = $screen ? $screen->id : '';

        if ( 'woocommerce_page_wc-settings' !== $screen_id ) {
            return;
        }

        if ( ! isset( $_GET['section'] ) || 'siha' !== $_GET['section'] ) {
            return;
        }

        // Enqueue admin styles
        wp_enqueue_style(
            'siha-admin-styles',
            SIHA_WC_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            SIHA_WC_VERSION
        );

        // Enqueue admin scripts
        wp_enqueue_script(
            'siha-admin-scripts',
            SIHA_WC_PLUGIN_URL . 'assets/js/admin.js',
            array( 'jquery' ),
            SIHA_WC_VERSION,
            true
        );

        // Localize script with AJAX URL and nonce
        wp_localize_script(
            'siha-admin-scripts',
            'sihaAdmin',
            array(
                'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
                'nonce'     => wp_create_nonce( 'siha_ajax_nonce' ),
                'i18n'      => array(
                    'syncStarted'   => __( 'Synchronization started...', 'siha-woocommerce' ),
                    'syncComplete'  => __( 'Synchronization complete!', 'siha-woocommerce' ),
                    'syncError'     => __( 'An error occurred during synchronization.', 'siha-woocommerce' ),
                ),
            )
        );
    }

    /**
     * Add admin notices.
     */
    public function admin_notices() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        // Check for required settings
        $api_key = get_option( 'woocommerce_siha_api_key' );
        if ( empty( $api_key ) ) {
            $settings_url = admin_url( 'admin.php?page=wc-settings&tab=products&section=siha' );
            echo '<div class="error"><p>';
            printf(
                /* translators: %s: Settings page URL */
                esc_html__( 'SIHA WooCommerce is almost ready. Please configure your %s.', 'siha-woocommerce' ),
                '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'API settings', 'siha-woocommerce' ) . '</a>'
            );
            echo '</p></div>';
        }
    }
}
