const { Shopify } = require('@shopify/shopify-api');
const { MongoDBAdapter } = require('@shopify/shopify-app-session-storage-mongodb');
const mongoose = require('mongoose');

/**
 * Sets up the session storage using MongoDB
 * @param {Object} server - Express server instance
 * @returns {Object} Session storage instance
 */
const setupSessions = (server) => {
  try {
    // Create a MongoDB session storage adapter
    const mongoSessionStorage = new MongoDBAdapter({
      url: process.env.MONGODB_URI,
      collection: 'sessions'
    });

    // Configure session storage
    const sessionConfig = {
      secret: process.env.SESSION_SECRET,
      resave: false,
      saveUninitialized: false,
      store: mongoSessionStorage,
      cookie: {
        secure: process.env.NODE_ENV === 'production',
        sameSite: 'none', // Required for embedded apps
        maxAge: 1000 * 60 * 60 * 24 * 7, // 1 week
        httpOnly: true,
        secure: true, // Required for cookies to work in embedded apps
      },
    };

    // Apply session middleware
    const { applyAuthMiddleware } = require('@shopify/shopify-app-express');
    const { auth, webhooks, verifyRequest } = applyAuthMiddleware({
      api: Shopify.API,
      auth: {
        path: '/api/auth',
        callbackPath: '/api/auth/callback',
        afterAuth: async (req, res, session) => {
          // Save the session to the database
          await mongoSessionStorage.storeSession(session);
          
          // Redirect to the app's home page after successful authentication
          const redirectUrl = `/?shop=${session.shop}&host=${req.query.host}`;
          res.redirect(redirectUrl);
        },
      },
      webhooks: {
        path: '/api/webhooks',
      },
      sessionStorage: mongoSessionStorage,
    });

    // Apply the middleware to the server
    server.use(auth);
    server.use(webhooks);
    server.use(verifyRequest);

    return mongoSessionStorage;
  } catch (error) {
    console.error('Error setting up sessions:', error);
    throw error;
  }
};

module.exports = { setupSessions };
