const { Shopify } = require('@shopify/shopify-api');

/**
 * Sets up authentication middleware for the Shopify app
 * @param {Object} server - Express server instance
 * @param {Object} sessionStorage - Session storage instance
 */
const setupAuth = (server, sessionStorage) => {
  try {
    // Apply Shopify authentication middleware
    const { applyAuthMiddleware } = require('@shopify/shopify-app-express');
    
    const { auth, webhooks, verifyRequest } = applyAuthMiddleware({
      api: Shopify.API,
      auth: {
        path: '/api/auth',
        callbackPath: '/api/auth/callback',
        afterAuth: async (req, res, session) => {
          // Store the session in the database
          await sessionStorage.storeSession(session);
          
          // Get the shop and host from the session
          const { shop, accessToken } = session;
          const host = req.query.host;
          
          // Register webhooks for the shop
          await registerWebhooks(shop, accessToken);
          
          // Redirect to the app's home page
          const redirectUrl = `/?shop=${shop}&host=${host}`;
          res.redirect(redirectUrl);
        },
      },
      webhooks: {
        path: '/api/webhooks',
      },
      sessionStorage,
    });

    // Apply the middleware to the server
    server.use(auth);
    server.use(webhooks);
    
    // Apply verifyRequest middleware to all routes except auth and webhooks
    server.use((req, res, next) => {
      if (req.path.startsWith('/api/auth') || req.path.startsWith('/api/webhooks')) {
        return next();
      }
      return verifyRequest()(req, res, next);
    });
    
  } catch (error) {
    console.error('Error setting up authentication:', error);
    throw error;
  }
};

/**
 * Registers required webhooks for the app
 * @param {string} shop - The shop domain
 * @param {string} accessToken - The access token for the shop
 */
const registerWebhooks = async (shop, accessToken) => {
  try {
    const client = new Shopify.Clients.Rest(shop, accessToken);
    
    // Define webhooks to register
    const webhooks = [
      {
        topic: 'ORDERS_CREATE',
        address: `${process.env.SHOPIFY_APP_URL}/api/webhooks/orders/create`,
        format: 'json',
      },
      {
        topic: 'PRODUCTS_UPDATE',
        address: `${process.env.SHOPIFY_APP_URL}/api/webhooks/products/update`,
        format: 'json',
      },
      {
        topic: 'INVENTORY_LEVELS_UPDATE',
        address: `${process.env.SHOPIFY_APP_URL}/api/webhooks/inventory/update`,
        format: 'json',
      },
    ];
    
    // Register each webhook
    for (const webhook of webhooks) {
      try {
        await client.post({
          path: 'webhooks',
          data: {
            webhook: {
              topic: webhook.topic,
              address: webhook.address,
              format: webhook.format,
            },
          },
        });
      } catch (error) {
        console.error(`Error registering ${webhook.topic} webhook:`, error.message);
      }
    }
  } catch (error) {
    console.error('Error in registerWebhooks:', error);
    throw error;
  }
};

module.exports = { setupAuth };
