require('dotenv').config();
const express = require('express');
const { join } = require('path');
const { createServer } = require('http');
const { parse } = require('url');
const next = require('next');
const { Shopify } = require('@shopify/shopify-api');
const { verifyRequest } = require('@shopify/shopify-app-express');
const mongoose = require('mongoose');
const winston = require('winston');
const helmet = require('helmet');
const cors = require('cors');
const compression = require('compression');
const morgan = require('morgan');
const bodyParser = require('body-parser');

// Import routes
const apiRoutes = require('./routes/api');
const webhookRoutes = require('./routes/webhooks');
const { setupAuth } = require('./middleware/auth');
const { setupSessions } = require('./middleware/sessions');
const { errorHandler } = require('./middleware/error');

// Initialize logger
const logger = winston.createLogger({
  level: 'info',
  format: winston.format.combine(
    winston.format.timestamp(),
    winston.format.json()
  ),
  transports: [
    new winston.transports.Console(),
    new winston.transports.File({ filename: 'error.log', level: 'error' }),
    new winston.transports.File({ filename: 'combined.log' })
  ]
});

// Initialize Next.js
const dev = process.env.NODE_ENV !== 'production';
const port = parseInt(process.env.PORT, 10) || 3000;
const app = next({ dev });
const handle = app.getRequestHandler();

// Connect to MongoDB
mongoose.connect(process.env.MONGODB_URI, {
  useNewUrlParser: true,
  useUnifiedTopology: true,
  serverSelectionTimeoutMS: 5000
}).then(() => {
  logger.info('Connected to MongoDB');
}).catch((err) => {
  logger.error('MongoDB connection error:', err);
  process.exit(1);
});

// Initialize Shopify context
const shopifyConfig = {
  apiKey: process.env.SHOPIFY_API_KEY,
  apiSecretKey: process.env.SHOPIFY_API_SECRET,
  scopes: process.env.SHOPIFY_SCOPES.split(','),
  hostName: process.env.SHOPIFY_APP_URL.replace(/^https?:\/\//, ''),
  hostScheme: process.env.SHOPIFY_APP_URL.split('://')[0],
  apiVersion: '2023-10',
  isEmbeddedApp: true,
  logger: {
    log: (severity, msg) => {
      switch (severity) {
        case 'error':
          logger.error(msg);
          break;
        case 'warning':
          logger.warn(msg);
          break;
        case 'info':
        default:
          logger.info(msg);
      }
    },
  },
};

const { Shopify: ShopifyContext } = Shopify;
Shopify.Context.initialize(shopifyConfig);

app.prepare().then(() => {
  const server = express();
  
  // Apply middleware
  server.use(helmet());
  server.use(cors());
  server.use(compression());
  server.use(morgan('dev'));
  server.use(bodyParser.json());
  server.use(bodyParser.urlencoded({ extended: true }));

  // Setup sessions and authentication
  const sessionStorage = setupSessions(server);
  setupAuth(server, sessionStorage);

  // API Routes
  server.use('/api', apiRoutes);
  
  // Webhook Routes
  server.use('/webhooks', webhookRoutes);

  // Handle Next.js pages
  server.all('*', (req, res) => {
    const parsedUrl = parse(req.url, true);
    handle(req, res, parsedUrl);
  });

  // Error handling
  server.use(errorHandler);

  // Start server
  server.listen(port, (err) => {
    if (err) throw err;
    logger.info(`> Ready on http://localhost:${port}`);
  });
}).catch((err) => {
  logger.error('Server error:', err);
  process.exit(1);
});
