/**
 * Custom API Error class
 * @class APIError
 * @extends Error
 */
class APIError extends Error {
  /**
   * Create a new APIError
   * @param {string} message - Error message
   * @param {number} statusCode - HTTP status code
   * @param {Array} errors - Array of error objects
   */
  constructor(message, statusCode = 500, errors = []) {
    super(message);
    this.name = this.constructor.name;
    this.statusCode = statusCode;
    this.errors = errors;
    
    // Maintain proper stack trace (only available in V8)
    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, this.constructor);
    }
  }
}

/**
 * 400 Bad Request Error
 */
class BadRequestError extends APIError {
  constructor(message = 'Bad Request', errors = []) {
    super(message, 400, errors);
  }
}

/**
 * 401 Unauthorized Error
 */
class UnauthorizedError extends APIError {
  constructor(message = 'Unauthorized', errors = []) {
    super(message, 401, errors);
  }
}

/**
 * 403 Forbidden Error
 */
class ForbiddenError extends APIError {
  constructor(message = 'Forbidden', errors = []) {
    super(message, 403, errors);
  }
}

/**
 * 404 Not Found Error
 */
class NotFoundError extends APIError {
  constructor(message = 'Not Found', errors = []) {
    super(message, 404, errors);
  }
}

/**
 * 409 Conflict Error
 */
class ConflictError extends APIError {
  constructor(message = 'Conflict', errors = []) {
    super(message, 409, errors);
  }
}

/**
 * 422 Unprocessable Entity Error
 */
class ValidationError extends APIError {
  constructor(message = 'Validation Error', errors = []) {
    super(message, 422, errors);
  }
}

/**
 * 429 Too Many Requests Error
 */
class RateLimitError extends APIError {
  constructor(message = 'Too Many Requests', errors = []) {
    super(message, 429, errors);
  }
}

/**
 * 500 Internal Server Error
 */
class InternalServerError extends APIError {
  constructor(message = 'Internal Server Error', errors = []) {
    super(message, 500, errors);
  }
}

/**
 * 503 Service Unavailable Error
 */
class ServiceUnavailableError extends APIError {
  constructor(message = 'Service Unavailable', errors = []) {
    super(message, 503, errors);
  }
}

module.exports = {
  APIError,
  BadRequestError,
  UnauthorizedError,
  ForbiddenError,
  NotFoundError,
  ConflictError,
  ValidationError,
  RateLimitError,
  InternalServerError,
  ServiceUnavailableError
};
