// server/services/sihaApiClient.js
const axios = require('axios')
const { APIError } = require('../utils/errors')

class SihaApiClient {
    constructor(apiKey) {
        this.client = axios.create({
            baseURL: process.env.SIHA_API_BASE_URL || 'https://api.siha.com/v1',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'Authorization': `Bearer ${apiKey}`,
            },
        })
    }

    /**
     * Make a request to the SIHA API
     * @private
     */
    async _request(method, endpoint, data = null, params = {}) {
        try {
            const response = await this.client({
                method,
                url: endpoint,
                data,
                params,
            })
            return response.data
        } catch (error) {
            if (error.response) {
                // The request was made and the server responded with a status code
                throw new APIError(
                    error.response.data?.message || 'SIHA API Error',
                    error.response.status,
                    error.response.data?.errors
                )
            } else if (error.request) {
                // The request was made but no response was received
                throw new APIError('No response received from SIHA API', 502)
            } else {
                // Something happened in setting up the request
                throw new APIError(`Error setting up request: ${error.message}`, 500)
            }
        }
    }

    // Inventory Methods

    /**
     * Get product inventory
     * @param {string} productId - The product ID
     * @returns {Promise<Object>} Product inventory data
     */
    async getProductInventory(productId) {
        return this._request('get', `/inventory/products/${productId}`)
    }

    /**
     * Update product inventory
     * @param {string} productId - The product ID
     * @param {Object} data - Inventory update data
     * @returns {Promise<Object>} Update response
     */
    async updateProductInventory(productId, data) {
        return this._request('put', `/inventory/products/${productId}`, data)
    }

    // Order Methods

    /**
     * Create a new order
     * @param {Object} orderData - Order data
     * @returns {Promise<Object>} Created order
     */
    async createOrder(orderData) {
        return this._request('post', '/deliveries/orders', orderData)
    }

    /**
     * Get order details
     * @param {string} orderId - The order ID
     * @returns {Promise<Object>} Order details
     */
    async getOrder(orderId) {
        return this._request('get', `/deliveries/orders/${orderId}`)
    }

    /**
     * Update order status
     * @param {string} orderId - The order ID
     * @param {string} status - New status
     * @param {string} [notes] - Optional notes
     * @returns {Promise<Object>} Update response
     */
    async updateOrderStatus(orderId, status, notes = '') {
        return this._request('patch', `/deliveries/orders/${orderId}/status`, {
            status,
            notes,
        })
    }

    // Product Methods

    /**
     * Get product details
     * @param {string} productId - The product ID
     * @returns {Promise<Object>} Product details
     */
    async getProduct(productId) {
        return this._request('get', `/inventory/products/${productId}`)
    }

    /**
     * Update product details
     * @param {string} productId - The product ID
     * @param {Object} data - Product data to update
     * @returns {Promise<Object>} Update response
     */
    async updateProduct(productId, data) {
        return this._request('put', `/inventory/products/${productId}`, data)
    }

    // Batch Operations

    /**
     * Sync multiple products inventory
     * @param {Array<Object>} products - Array of product updates
     * @returns {Promise<Object>} Batch update response
     */
    async batchUpdateInventory(products) {
        return this._request('post', '/inventory/batch-update', { products })
    }
}

module.exports = SihaApiClient
